﻿#pragma once
#include "Component.h"
#include "Align16.h"

namespace RTCam {

// Holds the transformation for an entity.
// NOTE: Non-uniform scaling is not fully tested
// PERF: See if the amount of SIMD vector loading/unloading can be reduced with refactoring.
class Transform : public Component, public Align16
{
public:
	Transform(const shared_ptr<Entity>& entity);
	~Transform(void);

	// Updates the constant buffer on this transform
	void UpdateWorldAndNormalMatrices();
	void UpdateWorldViewProjectionMatrix(CXMMATRIX viewProjection, bool resetCameraMotion);

	// World space transformations
	XMFLOAT3A GetForward() const;
	XMFLOAT3A GetRight() const;
	XMFLOAT3A GetUp() const;

	XMVECTOR GetForwardVector() const;
	XMVECTOR GetRightVector() const;
	XMVECTOR GetUpVector() const;

	XMFLOAT3A GetWorldPosition() const;
	XMFLOAT4A GetWorldRotation() const;
	XMFLOAT3A GetWorldScale() const;
	void SetWorldPosition(const XMFLOAT3A& position);
	void SetWorldRotation(const XMFLOAT4A& rotation);

	// Local transformations
	XMFLOAT3A GetLocalPosition() const;
	XMFLOAT3A GetLocalScale() const;
	XMFLOAT4A GetLocalRotation() const;
	void SetLocalPosition(const XMFLOAT3A& position);
	void SetLocalRotation(const XMFLOAT4A& rotation);
	void SetLocalScale(const XMFLOAT3A& scale);

	// World to local and vice versa
	XMFLOAT3A TransformPointLocalToWorld(const XMFLOAT3A& position);
	XMFLOAT3A TransformPointWorldToLocal(const XMFLOAT3A& position);

private:
	// Sets the flag that a transform matrix needs to be updated
	void InvalidateLocalTransformMatrix();
	void InvalidateGlobalTransformMatrix();

	XMMATRIX UpdateLocalTransform() const;
	XMMATRIX UpdateGlobalTransform() const;

	XMFLOAT4X4A GetLocalTransform() const;
	XMFLOAT4X4A GetGlobalTransform() const;

	XMMATRIX GetLocalTransformMatrix() const;
	XMMATRIX GetGlobalTransformMatrix() const;

	mutable bool m_localMatrixNeedsUpdate;
	mutable bool m_globalMatrixNeedsUpdate;
	mutable bool m_parentNeedsUpdate;
	mutable bool m_childrenNeedUpdate;

	XMFLOAT3A m_localPosition;
	XMFLOAT3A m_localScale;
	XMFLOAT4A m_localRotation; // Quaternion

	// Cached transformations
	mutable XMFLOAT3A m_cachedWorldPosition;
	mutable XMFLOAT3A m_cachedWorldScale;
	mutable XMFLOAT4A m_cachedWorldRotation;

	// Transformation matrices.
	mutable XMFLOAT4X4A m_cachedLocalTransformMatrix;
	mutable XMFLOAT4X4A m_cachedGlobalTransformMatrix;

};

} // end namespace